<?php

namespace App\Filament\Resources;

use Filament\Forms;
use Filament\Tables;
use App\Models\Absensi;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Filament\Facades\Filament;
use Illuminate\Support\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Tables\Columns\MapAbsen;
use Filament\Resources\Resource;
use Filament\Tables\Filters\Filter;
use App\Tables\Columns\MapAbsenPulang;
use Filament\Tables\Actions\BulkAction;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use App\Filament\Resources\AbsensiResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use App\Filament\Resources\AbsensiResource\RelationManagers;

class AbsensiResource extends Resource
{
    protected static ?string $model = Absensi::class;

    protected static ?string $navigationIcon = 'heroicon-o-presentation-chart-line';
    protected static ?string $navigationGroup = 'Penilaian Kinerja';

    public static function form(Form $form): Form
    {
        Filament::registerRenderHook(
            'head.end', // Tempatkan di akhir head agar script dijalankan
            fn() => view('filament.scripts.geolocation')
        );

        return $form
            ->schema([
                Forms\Components\Select::make('shift_id')
                    ->required()
                    ->relationship('shift', 'nama'),
                Forms\Components\Select::make('jenis_absen_id')
                    ->required()
                    ->relationship('jenisAbsen', 'nama'),
                Forms\Components\Select::make('user_id')
                    ->required()
                    ->relationship('user', 'name'),
                Forms\Components\TimePicker::make('jam_datang'),
                Forms\Components\TextInput::make('lokasi_datang')
                    ->label('Lokasi Datang')
                    ->extraAttributes([
                        'id' => 'lokasi_datang',
                        'readonly' => true,
                    ]),
                Forms\Components\TimePicker::make('jam_pulang'),
                Forms\Components\TextInput::make('lokasi_pulang')
                    ->maxLength(255),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(function () {
                return Absensi::orderBy('id', 'DESC');
            })
            ->columns([
                // Tables\Columns\TextColumn::make('id')
                //     ->label('NO')
                //     ->rowIndex()
                //     ->searchable(),
                Tables\Columns\TextColumn::make('shift.nama')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('jenisAbsen.nama')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('user.name')
                    ->numeric()
                    ->sortable(),

                Tables\Columns\TextColumn::make('jam_datang')
                    ->time(),
                MapAbsen::make('lokasi_datang')
                    ->label('Lokasi Datang'),

                Tables\Columns\TextColumn::make('jam_pulang'),
                MapAbsenPulang::make('lokasi_pulang')
                    ->label('Lokasi Pulang'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Tanggal')
                    // ->dateTime()
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //

                Filter::make('user_id')
                    ->label('Petugas')
                    ->form([
                        Forms\Components\Select::make('user_id')
                            ->relationship('user', 'name')
                            ->searchable()
                            ->preload(),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['user_id'],
                            fn($query, $user_id) => $query->where('user_id', $user_id)
                        );
                    }),
                Filter::make('created_from')
                    ->form([
                        DatePicker::make('created_from')
                            ->label('Tanggal Dari'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['created_from'],
                            fn($query, $date) => $query->whereDate('created_at', '>=', $date)
                        );
                    }),

                Filter::make('created_to')
                    ->form([
                        DatePicker::make('created_to')
                            ->label('Tanggal Sampai'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['created_to'],
                            fn($query, $date) => $query->whereDate('created_at', '<=', $date)
                        );
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                ExportBulkAction::make()
                    ->label('Eksport Excel Berdasarkan Yang Dipilih'),

                BulkAction::make('exportPdf')
                    ->label('Export ke PDF')
                    ->icon('heroicon-o-document')
                    ->action(function ($records) {
                        // Load relasi user, validator, dan jenis kerja
                        $data = $records->load(['user.bidang', 'jenis', 'uraian'])->toArray();
                        // Ambil rentang bulan dari data yang di-*bulk*
                        Carbon::setLocale('id');
                        $bulanAwal = Carbon::parse(min(array_column($data, 'created_at')))->translatedFormat('F Y');
                        $bulanAkhir = Carbon::parse(max(array_column($data, 'created_at')))->translatedFormat('F Y');

                        // Format judul PDF berdasarkan periode bulan
                        $judulPeriode = ($bulanAwal == $bulanAkhir) ? $bulanAwal : "$bulanAwal - $bulanAkhir";

                        // dd($judulPeriode);
                        // Buat PDF dalam mode landscape
                        // Generate PDF dengan periode bulan
                        $pdf = Pdf::loadView('exports.absensi', [
                            'data' => $data,
                            'judulPeriode' => $judulPeriode
                        ])->setPaper('a4', 'landscape');

                        // Nama file berdasarkan periode bulan
                        return response()->streamDownload(
                            fn() => print($pdf->output()),
                            "absensi_{$judulPeriode}.pdf"
                        );
                    }),
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListAbsensis::route('/'),
            // 'create' => Pages\CreateAbsensi::route('/create'),
            // 'edit' => Pages\EditAbsensi::route('/{record}/edit'),
        ];
    }
}
