<?php

namespace App\Filament\Resources;

use stdClass;
use Carbon\Carbon;
use Filament\Forms;
use Filament\Tables;
use App\Models\Kinerja;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Barryvdh\DomPDF\Facade\Pdf;
use Filament\Resources\Resource;
use Filament\Tables\Filters\Filter;
use App\Tables\Columns\MapAktivitas;
use Illuminate\Support\Facades\Auth;
use Filament\Tables\Actions\BulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Contracts\HasTable;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use App\Filament\Resources\KinerjaResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use App\Filament\Resources\KinerjaResource\RelationManagers;

class KinerjaResource extends Resource
{
    protected static ?string $model = Kinerja::class;

    protected static ?string $navigationIcon = 'heroicon-o-calendar';
    protected static ?string $navigationGroup = 'Penilaian Kinerja';
    protected static ?string $label = 'Validasi Aktivitas';

    public static function getNavigationBadge(): ?string
    {
        return Kinerja::where('status_validasi', 'pending')->count();
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('user_id')
                    ->required()
                    ->searchable()
                    ->preload()
                    ->disabled()
                    ->relationship('user', 'name'),
                Forms\Components\DatePicker::make('tanggal')
                    ->required()
                    ->disabled(),
                Forms\Components\TimePicker::make('jam_mulai')
                    ->required()
                    ->disabled(),
                Forms\Components\TimePicker::make('jam_selesai')
                    ->required()
                    ->disabled(),
                Forms\Components\Select::make('jenis_kerja_id')
                    ->label('Aktivitas')
                    ->preload()
                    ->searchable()
                    ->disabled()
                    ->required()
                    ->allowHtml()
                    ->relationship('jenisKerja', 'uraian'),
                Forms\Components\TextInput::make('aktivitas_lain')
                    ->maxLength(255)
                    ->disabled(),
                Forms\Components\Textarea::make('keterangan')
                    ->columnSpanFull()
                    ->disabled(),
                Forms\Components\FileUpload::make('bukti_foto')
                    ->directory('aktivitas')
                    ->columnSpanFull(),
                Forms\Components\TextInput::make('latitude')
                    ->numeric()
                    ->disabled(),
                Forms\Components\TextInput::make('longitude')
                    ->numeric()
                    ->disabled(),
                Forms\Components\Select::make('validator_id')
                    ->relationship('validator', 'name')
                    ->searchable()
                    ->afterStateHydrated(function ($state, $set) {
                        $set('validator_id', Auth::id());
                    }),
                Forms\Components\Select::make('status_validasi')
                    ->options([
                        'pending' => 'pending',
                        'disetujui' => 'disetujui',
                        'ditolak' => 'ditolak',
                    ])
                    ->searchable()
                    ->preload()
                    ->required(),
                Forms\Components\RichEditor::make('keterangan_validasi')
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(function () {
                return Kinerja::query()->orderBy('id', 'Desc');
            })

            ->columns([
                // Tables\Columns\TextColumn::make('id')
                //     ->label('NO')
                //     ->rowIndex()
                //     ->searchable(),
                Tables\Columns\TextColumn::make('status_validasi')
                    ->label('Status')
                    ->searchable()
                    ->badge()
                    ->colors([
                        'warning' => 'pending',
                        'success' => 'disetujui',
                        'danger' => 'ditolak',
                    ]),
                Tables\Columns\TextColumn::make('user.name')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('tanggal')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('jam_mulai'),
                Tables\Columns\TextColumn::make('jam_selesai'),
                Tables\Columns\TextColumn::make('jenisKerja.uraian')
                    ->label('Aktivitas')
                    ->html()
                    ->searchable(),
                Tables\Columns\TextColumn::make('aktivitas_lain')
                    ->searchable(),
                Tables\Columns\ImageColumn::make('bukti_foto')
                    ->searchable(),
                // Tables\Columns\ViewColumn::make('bukti_foto')
                //     ->view('components.button-preview'),

                MapAktivitas::make('map')
                    ->label('Lokasi '),
                Tables\Columns\TextColumn::make('validator.name')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('keterangan_validasi')
                    ->searchable()
                    ->html()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Di Input')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //

                Filter::make('status_validasi')
                    ->label('status_validasi')
                    ->form([
                        Forms\Components\Select::make('status_validasi')
                            ->options(
                                Kinerja::query()
                                    ->select('status_validasi')
                                    ->distinct()
                                    ->pluck('status_validasi', 'status_validasi')
                                    ->toArray()
                            )
                            ->searchable()
                            ->preload()
                            ->placeholder('Pilih Status Yang Tersedia'), // Placeholder opsional
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['status_validasi'] ?? null,
                            fn($query, $status_validasi) => $query->where('status_validasi', $status_validasi)
                        );
                    }),
                Filter::make('user_id')
                    ->label('Petugas')
                    ->form([
                        Forms\Components\Select::make('user_id')
                            ->relationship('user', 'name')
                            ->searchable()
                            ->preload(),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['user_id'],
                            fn($query, $user_id) => $query->where('user_id', $user_id)
                        );
                    }),
                Filter::make('created_from')
                    ->form([
                        DatePicker::make('created_from')
                            ->label('Tanggal Dari'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['created_from'],
                            fn($query, $date) => $query->whereDate('created_at', '>=', $date)
                        );
                    }),

                Filter::make('created_to')
                    ->form([
                        DatePicker::make('created_to')
                            ->label('Tanggal Sampai'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['created_to'],
                            fn($query, $date) => $query->whereDate('created_at', '<=', $date)
                        );
                    }),
                Filter::make('bulan')
                    ->form([
                        Forms\Components\Select::make('bulan')
                            ->label('Pilih Bulan')
                            ->options([
                                '1' => 'Januari',
                                '2' => 'Februari',
                                '3' => 'Maret',
                                '4' => 'April',
                                '5' => 'Mei',
                                '6' => 'Juni',
                                '7' => 'Juli',
                                '8' => 'Agustus',
                                '9' => 'September',
                                '10' => 'Oktober',
                                '11' => 'November',
                                '12' => 'Desember',
                            ]),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['bulan'],
                            fn($query, $bulan) => $query->whereMonth('created_at', '=', $bulan)
                        );
                    }),
                Filter::make('tahun')
                    ->form([
                        Forms\Components\Select::make('tahun')
                            ->label('Pilih Tahun')
                            ->options(
                                collect(range(date('Y') - 10, date('Y')))
                                    ->mapWithKeys(fn($year) => [$year => $year])
                                    ->toArray()
                            ),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['tahun'],
                            fn($query, $tahun) => $query->whereYear('created_at', '=', $tahun)
                        );
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()->label('Validasi'),
            ])
            ->bulkActions([
                ExportBulkAction::make()
                    ->label('Eksport ke Excel')
                    ->icon('heroicon-o-document-arrow-down'),
                BulkAction::make('exportPdf')
                    ->label('Export ke PDF')
                    ->icon('heroicon-o-document')
                    ->action(function ($records) {
                        // Load relasi user, validator, dan jenis kerja
                        $data = $records->load(['user.bidang', 'validator', 'jenis'])->toArray();
                        // Ambil rentang bulan dari data yang di-*bulk*
                        Carbon::setLocale('id');
                        $bulanAwal = Carbon::parse(min(array_column($data, 'tanggal')))->translatedFormat('F Y');
                        $bulanAkhir = Carbon::parse(max(array_column($data, 'tanggal')))->translatedFormat('F Y');

                        // Format judul PDF berdasarkan periode bulan
                        $judulPeriode = ($bulanAwal == $bulanAkhir) ? $bulanAwal : "$bulanAwal - $bulanAkhir";

                        // dd($judulPeriode);
                        // Buat PDF dalam mode landscape
                        // Generate PDF dengan periode bulan
                        $pdf = Pdf::loadView('exports.kinerja', [
                            'data' => $data,
                            'judulPeriode' => $judulPeriode
                        ])->setPaper('a4', 'landscape');

                        // Nama file berdasarkan periode bulan
                        return response()->streamDownload(
                            fn() => print($pdf->output()),
                            "kinerja_{$judulPeriode}.pdf"
                        );
                    }),
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListKinerjas::route('/'),
            // 'create' => Pages\CreateKinerja::route('/create'),
            // 'edit' => Pages\EditKinerja::route('/{record}/edit'),
        ];
    }
}
